<?php
require 'auth.php';
session_start();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $_SESSION['status'] = [
        'type' => 'error',
        'msg'  => 'Invalid request method'
    ];
    header('Location: composer.php');
    exit;
}

// ==============================
// INPUT
// ==============================
$from_email = trim($_POST['from_email'] ?? '');
$from_name  = trim($_POST['from_name'] ?? '');
$reply_to   = trim($_POST['reply_to'] ?? $from_email);
$to         = trim($_POST['to'] ?? '');
$cc         = trim($_POST['cc'] ?? '');
$bcc        = trim($_POST['bcc'] ?? '');
$subject    = trim($_POST['subject'] ?? '');
$html_body  = $_POST['html_content'] ?? '';

// ==============================
// BASIC VALIDATION
// ==============================
if (
    empty($from_email) ||
    empty($to) ||
    empty($subject) ||
    empty($html_body)
) {
    $_SESSION['status'] = [
        'type' => 'error',
        'msg'  => 'Please fill in all required fields'
    ];
    header('Location: composer.php');
    exit;
}

// Validate email format only (NO domain restriction)
if (!filter_var($from_email, FILTER_VALIDATE_EMAIL)) {
    $_SESSION['status'] = [
        'type' => 'error',
        'msg'  => 'Invalid From email address'
    ];
    header('Location: composer.php');
    exit;
}

// ==============================
// ATTACHMENT STORAGE SETUP
// ==============================
$attachmentsBase = __DIR__ . '/attachments';
if (!is_dir($attachmentsBase)) {
    mkdir($attachmentsBase, 0755, true);
}

$savedAttachments = [];
$emailFolder = date('Y-m-d_H-i-s') . '_' . $_SESSION['user'];
$emailPath = $attachmentsBase . '/' . $emailFolder;

// ==============================
// EMAIL HEADERS & BOUNDARY
// ==============================
$boundary = md5(uniqid(time(), true));

$headers  = "From: {$from_name} <{$from_email}>\r\n";
$headers .= "Reply-To: {$reply_to}\r\n";

if (!empty($cc)) {
    $headers .= "Cc: {$cc}\r\n";
}
if (!empty($bcc)) {
    $headers .= "Bcc: {$bcc}\r\n";
}

$headers .= "MIME-Version: 1.0\r\n";
$headers .= "Content-Type: multipart/mixed; boundary=\"{$boundary}\"\r\n";

// ==============================
// MESSAGE BODY (HTML)
// ==============================
$message  = "--{$boundary}\r\n";
$message .= "Content-Type: text/html; charset=UTF-8\r\n";
$message .= "Content-Transfer-Encoding: 7bit\r\n\r\n";
$message .= $html_body . "\r\n";

// ==============================
// ATTACHMENTS (SEND + SAVE)
// ==============================
if (!empty($_FILES['attachments']['name'][0])) {

    mkdir($emailPath, 0755, true);

    for ($i = 0; $i < count($_FILES['attachments']['name']); $i++) {

        if ($_FILES['attachments']['error'][$i] === UPLOAD_ERR_OK) {

            $file_tmp  = $_FILES['attachments']['tmp_name'][$i];
            $file_name = basename($_FILES['attachments']['name'][$i]);

            // SAVE FILE FOR FUTURE VIEWING
            $target = $emailPath . '/' . $file_name;
            if (move_uploaded_file($file_tmp, $target)) {
                $savedAttachments[] = [
                    'name' => $file_name,
                    'path' => 'attachments/' . $emailFolder . '/' . $file_name
                ];
            }

            // ATTACH TO EMAIL
            $file_data = chunk_split(base64_encode(
                file_get_contents($target)
            ));

            $message .= "--{$boundary}\r\n";
            $message .= "Content-Type: application/octet-stream; name=\"{$file_name}\"\r\n";
            $message .= "Content-Disposition: attachment; filename=\"{$file_name}\"\r\n";
            $message .= "Content-Transfer-Encoding: base64\r\n\r\n";
            $message .= $file_data . "\r\n";
        }
    }
}

$message .= "--{$boundary}--";

// ==============================
// SEND EMAIL
// ==============================
$sent = mail($to, $subject, $message, $headers);

// ==============================
// LOG RESULT
// ==============================
if ($sent) {

    $logFile = __DIR__ . '/logs/emails.json';

    $logs = [];
    if (file_exists($logFile)) {
        $logs = json_decode(file_get_contents($logFile), true) ?: [];
    }

    $logs[] = [
        'id'          => uniqid('mail_', true), // REQUIRED FOR EDIT FEATURE
        'date'        => date('Y-m-d H:i:s'),
        'user'        => $_SESSION['user'],
        'from'        => $from_email,
        'to'          => $to,
        'cc'          => $cc,
        'bcc'         => $bcc,
        'subject'     => $subject,
        'body'        => $html_body,
        'attachments' => $savedAttachments
    ];

    file_put_contents(
        $logFile,
        json_encode($logs, JSON_PRETTY_PRINT)
    );

    $_SESSION['status'] = [
        'type' => 'success',
        'msg'  => 'Email sent successfully'
    ];

} else {
    $_SESSION['status'] = [
        'type' => 'error',
        'msg'  => 'Failed to send email'
    ];
}

// ==============================
// REDIRECT BACK
// ==============================
header('Location: composer.php');
exit;
